/*
 * Written by Dawid Kurzyniec and released to the public domain, as explained
 * at http://creativecommons.org/licenses/publicdomain
 */

package edu.emory.mathcs.util.collections.longs;

import java.util.*;

/**
 * Map from long keys to object values.
 *
 * @author Dawid Kurzyniec
 * @version 1.0
 */
public interface LongMap {
    /**
     * Returns the number of entries in this map.
     * @return the number of entries in this map.
     */
    int size();

    /**
     * Returns true if this map is empty; false otherwise.
     * @return true if this map is empty; false otherwise.
     */
    boolean isEmpty();

    /**
     * Returns true if this map contains mapping for the specified number;
     * false otherwise.
     *
     * @return true if this map contains mapping for the specified number;
     *         false otherwise.
     */
    boolean containsKey(long key);

    /**
     * Returns true if this map contains mapping with the specified value;
     * false otherwise. Usually, this method has a linear complexity.
     *
     * @return true if this map contains mapping with the specified value;
     *         false otherwise.
     */
    boolean containsValue(Object value);

    /**
     * Retrieves the mapping for the specified number, or null if none.
     * @param key the search key
     * @return the mapping for the specified number, or null if none.
     */
    Object get(long key);

    /**
     * Maps specified value to the given long key, and returns the
     * value previously mapped for that key, or null if there were no mapping
     * for that key before.
     * @param key the key
     * @param value the value
     *
     * @return the value previously mapped to key, or null if none
     */
    Object put(long key, Object value);

    /**
     * Removes the mapping for the given key and returns the removed value,
     * or null if there was no mapping for that key.
     *
     * @param key the key to remove the mapping for
     * @return the value mapped to the key
     */
    Object remove(long key);

    /**
     * Puts all mappings from the specified map into this map.
     * @param map the map to copy the mappings from.
     */
    void putAll(LongMap map);

    /**
     * Removes all the mappings.
     */
    void clear();

    /**
     * Returns the view of keys in this map, as a LongSet. The view is backed
     * by this map. Supports removals.
     * @return the key set view of this map
     */
    LongSet keySet();

    /**
     * Returns the read-only view of values in this map, as a Collection.
     * The view is backed by this map.
     * @return the value view of this map
     */
    Collection values();

    /**
     * Returns the view of entries in this map, as a Set of LongMap.Entry
     * elements. The view is backed by this map. Supports removals, and
     * value changes via entry.setValue().
     *
     * @return the entry set view of this map
     */
    Set entrySet();

    /**
     * Represents the mapping from a long key to an object value. Returned by
     * entrySet().iterator(). The returned entry becomes invalid (i.e. its
     * behavior is no longer guaranteed) if the map is modified in a way other
     * than via iterator.remove() or entry.setValue().
     */
    public interface Entry {
        /** Returns the key associated with this entry. */
        long getKey();
        /** Returns the value associated with this entry. */
        Object getValue();
        /** Assigns a new value to the given key, and returns the old value. */
        Object setValue(Object value);

        boolean equals(Object o);
        int hashCode();
    }

    boolean equals(Object o);
    int hashCode();
}
